﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Reflection;
using System.Text;

using Newtonsoft.Json;

using VIRP.EFR.BO;

namespace VIRP.EFR.BLL
{
    public static class JsonSchemaValidator
    {
        private static Dictionary<Type, PropertyInfo[]> SupportedTypes;

        private const bool EnforceMissingMemberHandling = false;
        private const bool IgnoreNullValues = true;

        static JsonSchemaValidator()
        {
            SupportedTypes = new Dictionary<Type, PropertyInfo[]>
            {
                { typeof(StudyQueryResponse), Type.GetType(typeof(StudyQueryResponse).ToString()).GetProperties() }
            };
        }

        public static T DeserializeAndValidate<T>(string json)
        {
            T output = default(T);

            var validated = Validate<T>(json);
            if (!string.IsNullOrEmpty(validated))
            {
                byte[] jsonBytes = Encoding.UTF8.GetBytes(json);
                using (var stream = new MemoryStream(jsonBytes))
                {
                    output = Deserialize<T>(stream);
                }
            }

            return output;
        }

        private static string Validate<T>(string json)
        {
            if (string.IsNullOrEmpty(json))
                throw new Exception("Invalid Json: input string is null or empty.");

            ValidateType(typeof(T));

            return ValidateJson<T>(json);
        }

        private static void ValidateType(Type type)
        {
            if (!SupportedTypes.ContainsKey(type))
                throw new Exception(string.Format("Type {0} not supported.", type.ToString()));
        }

        private static string ValidateJson<T>(string input)
        {
            foreach (var p in SupportedTypes[typeof(T)])
            {
                if (!input.Contains(p.Name, StringComparison.OrdinalIgnoreCase))
                    throw new Exception(string.Format("Properties of type: {0} do not match expected names.", typeof(T).ToString()));
            }

            return input;
        }

        private static TResult Deserialize<TResult>(Stream responseStream)
        {
            using (var sr = new StreamReader(responseStream))
            {
                using (var reader = new JsonTextReader(sr))
                {
                    var serializer = new JsonSerializer
                    {
                        MissingMemberHandling = EnforceMissingMemberHandling ? MissingMemberHandling.Error : MissingMemberHandling.Ignore,
                        NullValueHandling = IgnoreNullValues ? NullValueHandling.Ignore : NullValueHandling.Include
                    };

                    return serializer.Deserialize<TResult>(reader);
                }
            }
        }
    }
}